<?php
/**
 * User Set Status
 *
 * @package     AutomatorWP\Integrations\Autonami\Triggers\User_Set_Status
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Autonami_User_Set_Status extends AutomatorWP_Integration_Trigger {

    public $integration = 'autonami';
    public $trigger = 'autonami_user_set_status';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User status is set to a specific status', 'automatorwp-pro' ),
            'select_option'     => __( 'User status is set to a specific <strong>status</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Status. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User status is set to %1$s %2$s time(s)', 'automatorwp-pro' ), '{status}', '{times}' ),
            /* translators: %1$s: Status. */
            'log_label'         => sprintf( __( 'User status is set to %1$s', 'automatorwp-pro' ), '{status}' ),
            'action'            => array(
                'bwfcrm_after_contact_subscribed',
                'bwfcrm_after_contact_unsubscribed',
            ),
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'status' => array(
                    'from' => 'status',
                    'fields' => array(
                        'status' => array(
                            'name' => __( 'Status:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'any'           => __( 'any status', 'automatorwp-pro' ),
                                'subscribed'    => __( 'Subscribed', 'automatorwp-pro' ),
                                'unsubscribed'  => __( 'Unsubscribed', 'automatorwp-pro' ),
                            ),
                            'default' => 'any'
                        ),
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_autonami_status_tags(),
                automatorwp_autonami_email_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param WooFunnels_Contact|array $contact
     */
    public function listener( $contact ) {

        // Get contact email
        if ( is_object ( $contact ) ) {
            $email = $contact->email;
        } else {
            $email = $contact[0]['recipient'];
        }
        
		$user = get_user_by( 'email', $email );

        // Make sure contact has an user ID assigned
        if ( $user->ID === 0 ) {
            return;
        }

        $contact_obj = new BWFCRM_Contact( $email );

        $status = $contact_obj->check_contact_unsubscribed();

        if ( empty ( $status ) ) {
            $status = 'subscribed';
        } else {
            $status = 'unsubscribed';
        }

        // Trigger the status change
        automatorwp_trigger_event( array(
            'trigger'           => $this->trigger,
            'user_id'           => $user->ID,
            'status'            => $status,
            'contact_email'     => $email,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if status is not received
        if( ! isset( $event['status'] ) ) {
            return false;
        }

        // Don't deserve if status doesn't match with the trigger option
        if( $trigger_options['status'] !== 'any' && $trigger_options['status'] !== $event['status'] ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['contact_email'] = ( isset( $event['contact_email'] ) ? $event['contact_email'] : '' );
        $log_meta['status'] = ( isset( $event['status'] ) ? $event['status'] : '' );

        return $log_meta;

    }

}

new AutomatorWP_Autonami_User_Set_Status();